// vim: tabstop=4:shiftwidth=4:noexpandtab
/***************************************************************************
* Copyright © 2015 Ken Moffat <ken at linuxfromscratch dot org>
*
* Derived from the maldives theme which is
* Copyright (c) 2013 Abdurrahman AVCI <abdurrahmanavci@gmail.com>
*
* Permission is hereby granted, free of charge, to any person
* obtaining a copy of this software and associated documentation
* files (the "Software"), to deal in the Software without restriction,
* including without limitation the rights to use, copy, modify, merge,
* publish, distribute, sublicense, and/or sell copies of the Software,
* and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
* OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
* OR OTHER DEALINGS IN THE SOFTWARE.
*
***************************************************************************/

import QtQuick 2.0
import SddmComponents 2.0

Rectangle {
	id: container
	width: 640
	height: 480

	TextConstants { id: textConstants }

	Connections {
		target: sddm

		onLoginSucceeded: {
			errorMessage.color = "steelblue"
			errorMessage.text = textConstants.loginSucceeded
		}

		onLoginFailed: {
			errorMessage.color = "red"
			errorMessage.text = textConstants.loginFailed
		}
	}

	Repeater {
		model: screenModel
		Background {
			x: geometry.x; y: geometry.y; width: geometry.width; height:geometry.height
			source: config.background
			fillMode: Image.PreserveAspectCrop
			onStatusChanged: {
				if (status == Image.Error && source != config.defaultBackground) {
					source = config.defaultBackground
				}
			}
		}
	}

	Rectangle {
		property variant geometry: screenModel.geometry(screenModel.primary)
		x: geometry.x; y: geometry.y; width: geometry.width; height: geometry.height
		color: "transparent"

		Clock {
			id: clock
			anchors.margins: 50
			anchors.top: parent.top; anchors.left: parent.left

			color: "cyan"
			timeFont.family: "Sans"
		}

		Rectangle {
			anchors.right: parent.right
			anchors.top: parent.top
			anchors.margins: 5
			width: 50 // without width nothing appears, but height seems to be unnecessary
			color: "transparent"

			Column {
				id: buttons
				width: parent.width
				height: parent.height

				ImageButton {
					id: loginButton
					source: "triangle.png"

					visible: true

					onClicked: sddm.login(name.text, password.text, session.index)

					KeyNavigation.backtab: session; KeyNavigation.tab: shutdownButton
				}

				ImageButton {
					id: shutdownButton
					source: "onoff4.png"

					visible: sddm.canPowerOff // text buttons are always visible!

					onClicked: sddm.powerOff()

					KeyNavigation.backtab: loginButton; KeyNavigation.tab: rebootButton
				}

				ImageButton {
					id: rebootButton
					source: "ringarrow.png"

					visible: sddm.canReboot

					onClicked: sddm.reboot()

					KeyNavigation.backtab: shutdownButton; KeyNavigation.tab: suspendButton
				}

				ImageButton {
					id: suspendButton
					source: "suspend4.png"

					visible: sddm.canSuspend

					onClicked: sddm.suspend()

					KeyNavigation.backtab: rebootButton; KeyNavigation.tab: name
				}

			}
		}

		Image {
			id: rectangle
			anchors.centerIn: parent
			width: Math.max(320, mainColumn.implicitWidth + 50)
			height: Math.max(320, mainColumn.implicitHeight + 50)

			source: "rectangle.png"

			Column {
				id: mainColumn
				anchors.centerIn: parent

				Text {
					anchors.horizontalCenter: parent.horizontalCenter
					color: "black"
					verticalAlignment: Text.AlignVCenter
					height: text.implicitHeight
					width: parent.width
					text: textConstants.welcomeText.arg(sddm.hostName)
					wrapMode: Text.WordWrap
					font.pixelSize: 24
					elide: Text.ElideRight
					horizontalAlignment: Text.AlignHCenter
				}

				Text {
					id: errorMessage
					anchors.horizontalCenter: parent.horizontalCenter
					text: textConstants.prompt
					font.pixelSize: 16
				}

				Text {
					id: lblName
					width: parent.width
					text: textConstants.userName
					font.bold: true
					font.pixelSize: 16
				}

				TextBox {
					id: name
					width: 150; height: 30
					text: userModel.lastUser
					font.pixelSize: 14

					KeyNavigation.backtab: rebootButton; KeyNavigation.tab: password

					Keys.onPressed: {
						if (event.key === Qt.Key_Return || event.key === Qt.Key_Enter) {
							sddm.login(name.text, password.text, session.index)
							event.accepted = true
						}
					}
				}

				Text {
					id: lblPassword
					width: parent.width
					text: textConstants.password
					font.bold: true
					font.pixelSize: 16
				}

				PasswordBox {
					id: password
					width: 150; height: 30
					font.pixelSize: 14
					KeyNavigation.backtab: name; KeyNavigation.tab: session
					Keys.onPressed: {
						if (event.key === Qt.Key_Return || event.key === Qt.Key_Enter) {
							sddm.login(name.text, password.text, session.index)
							event.accepted = true
						}
					}
				}

				Text {
					id: lblSession
					width: parent.width
					text: textConstants.session
					wrapMode: TextEdit.WordWrap
					font.bold: true
					font.pixelSize: 16
				}

				ComboBox {
					id: session
					width: 100; height: 30
					font.pixelSize: 14

					arrowIcon: "angle-down.png"

					model: sessionModel
					index: sessionModel.lastIndex

					KeyNavigation.backtab: password; KeyNavigation.tab: name
				}

            }

        }
    }


	Component.onCompleted: {
		if (name.text == "")
			name.focus = true
		else
			password.focus = true
	}
}
