/* Copyright (c) 2004 Matthias S. Benkmann <article AT winterdrache DOT de>
   You may do everything with this code except misrepresent its origin.
   PROVIDED `AS IS' WITH ABSOLUTELY NO WARRANTY OF ANY KIND!


  pre-init is a temporary init for booting a Linux system whose filesystem
  root is not the root of the partition. pre-init's main task is to change
  the filesystem root using chroot(2) and then to replace itself with the 
  real init from the system to be booted. In addition pre-init performs some
  preparations that are necessary to make sure that the real init is started
  with valid standard input/output/error and to allow / to be remounted 
  properly.
  
  You can change pre-init's behaviour with the following arguments on the
  kernel command line (the "kernel ..." line in GRUB's menu.lst):
  
    lfsdir=<dir> : Override the directory that contains the LFS system (i.e.
                   the directory to chroot into, unless nochroot is given). 
                   Default is determined by stripping "/sbin/..." from the 
                   path where the pre-init binary is located (i.e. the path 
                   you specified in the "init=..." argument on the kernel 
                   command line).
  
    realinit=<path> : Override program to exec after chroot. Default is
                      "/sbin/init". You may prefix <path> with <lfsdir>, i.e.
                      "/sbin/init" and "<lfsdir>/sbin/init" are equivalent.
    
    makedev : Mount a ramfs on <lfsdir>/dev and create device nodes
              /dev/console, /dev/tty1 and /dev/null in it.
              This is useful if your root filesystem doesn't support Linux
              device nodes or some other reason keeps you from creating them
              on the disk.
              Note that these devices are created with incorrect permissions
              (only accessible by root). You will probably need to correct this
              in a boot script (the LFS boot scripts already do this).
    
    nochroot : Do not chroot into <lfsdir>. Do not use this unless you know
               what you're doing!
    
    bindmount=<list> : Uses bind-mounts to remap directories. <list> is a
                       comma-separated list of entries of the form dir1=dir2.
                       The list is processed from left to right and each entry
                       causes its dir1 to become a copy of its dir2.
                       If a dir does not start with a "/", it will be
                       relative to <lfsdir>.
*/

#include <unistd.h>
#include <string.h>
#include <sys/mount.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>

#define DEV_CONSOLE_MAJOR 5
#define DEV_CONSOLE_MINOR 1

void error(const char* msg);
void warning(const char* msg);
char* concat(const char* str1,...);
int is_console(int fd);
void do_bindmounts(const char* lfsdir, char* bindmount);

/* 
  Do not allow glibc to sabotage pre-init. 
*/
void __libc_check_standard_fds (void) {}

int main(int argc, char* argv[])
{
  if (getpid() != 1) _exit(1);
    
  /*
    Precaution: If our std* do not refer to /dev/console, get rid of them asap.
  */
  int weird_stdfds = 0;
  if (! (is_console(0) && is_console(1) && is_console(2) ) )
  {
    close(0); close(1); close(2);
    weird_stdfds = 1;
  }

  /*
    Process boot parameters.

    The kernel transforms ...=... arguments into environment variables, so
    check the environment for our parameters.
  */
  char* lfsdir = NULL; 
  char* envstr = getenv("lfsdir");
  if (envstr != NULL) lfsdir = envstr;
  unsetenv("lfsdir");
  
  char* realinit = "/sbin/init";
  envstr = getenv("realinit");
  if (envstr != NULL) realinit = envstr;
  unsetenv("realinit");

  int lfsmakedev = 0;
  envstr = getenv("makedev");
  if (envstr != NULL) lfsmakedev = 1;
  unsetenv("makedev");
  
  int nochroot = 0;
  envstr = getenv("nochroot");
  if (envstr != NULL) nochroot = 1;
  unsetenv("nochroot");
  
  char* bindmount = NULL; 
  envstr = getenv("bindmount");
  if (envstr != NULL) bindmount = envstr;
  unsetenv("bindmount");
  
  /*
    If pre-init is not called directly by the kernel, chances are that the
    caller wants to pass us parameters as command line arguments, so check
    these, too.
  */
  for (int i=1; i<argc; ++i)
  {
    if (strstr(argv[i],"lfsdir=") == argv[i])
      lfsdir = argv[i] + strlen("lfsdir=");
    else
    if (strstr(argv[i],"realinit=") == argv[i])
      realinit = argv[i] + strlen("realinit=");
    else
    if (strcmp(argv[i],"makedev") == 0)
      lfsmakedev = 1;
    else
    if (strcmp(argv[i],"nochroot") == 0)
      nochroot = 1;
    else
    if (strstr(argv[i],"bindmount=") == argv[i])
      bindmount = argv[i] + strlen("bindmount=");
    else
      continue;
      
    /*
      If we've processed a command line argument, remove it from the
      command line, so that we don't pass it on to the real init.
    */
    for (int j=i+1; j<argc; ++j) argv[j-1] = argv[j];
    --argc;
    --i;
    argv[argc]=NULL;
  }
  
  /*
    Autodetection of lfsdir if it was not passed on the command line.
  */
  if (lfsdir == NULL)
  {
    lfsdir = strdup(argv[0]);
    char* eod = strstr(lfsdir,"/sbin/");
    if (eod == NULL) 
      error(concat("Cannot derive LFS directory from argv[0] (",argv[0],"). Use lfsdir=<dir> boot parameter.",NULL));
    else
      *eod=0;
  }
  
  /*
    Bind lfsdir to itself. This trick allows us to access the root filesystem
    from inside chroot.
  */
  if (!nochroot)
  {
    if (mount(lfsdir, lfsdir, "none", MS_BIND|MS_MGC_VAL, NULL) < 0)
      error(concat("Binding LFS directory (",lfsdir,") to itself failed! (",strerror(errno),")",NULL));
  }
  
  /* Set up <lfsdir>/dev if requested. */
  if (lfsmakedev)
  {
    if (mount("ramfs", concat(lfsdir,"/dev",NULL), "ramfs", MS_MGC_VAL, NULL) < 0)
      error(concat("Mounting ramfs on ",lfsdir,"/dev failed! (",strerror(errno),")",NULL));
    
    if (
            (mknod(concat(lfsdir,"/dev/console",NULL), 0600|S_IFCHR , makedev(5,1)) < 0)
         || (mknod(concat(lfsdir,"/dev/null",NULL), 0600|S_IFCHR , makedev(1,3)) < 0)
         || (mknod(concat(lfsdir,"/dev/tty1",NULL), 0600|S_IFCHR , makedev(4,1)) < 0)
       )
      error(concat("Creating console, null or tty1 failed! (",strerror(errno),")",NULL));  
  }
  
  /* 
    Point standard file descriptors to <lfsdir>/dev/console.
  */
  int fd = open(concat(lfsdir,"/dev/console",NULL), O_RDWR);
  if (fd >= 0)
  {
    dup2(fd, 0);
    dup2(fd, 1);
    dup2(fd, 2);
  }
  else
    warning(concat("Could not point standard file descriptors to ",lfsdir,"/dev/console! (",strerror(errno),")",NULL));

  /* Close fd if it's not one of the standard descriptors. */
  if (fd>2) close(fd);
  
  /*
    Now that we (hopefully) have a good stdout, output warning if we were
    started with weird std* fds.
  */
  if (weird_stdfds)
    warning("Did not get console device as stdin/stdout/stderr at startup!");
  
  /*
    Perform bind-mounts if requested
  */
  if (bindmount != NULL)
    do_bindmounts(lfsdir, bindmount);
  
  /*
    Change directory to <lfsdir> in preparation for chroot.
  */
  if (!nochroot)
  {
    if (chdir(lfsdir) < 0)
      error(concat("Cannot chdir('",lfsdir,"')! (",strerror(errno),")",NULL));
  }
  
  /*
    Chroot into <lfsdir>.
  */
  if (!nochroot)
  {
    if (chroot(lfsdir) < 0)
      error(concat("Cannot chroot('",lfsdir,"')!  (",strerror(errno),")",NULL));  
  }

  /* 
    Point standard file descriptors to new /dev/console.
  */
  fd = open("/dev/console", O_RDWR);
  if (fd >= 0)
  {
    dup2(fd, 0);
    dup2(fd, 1);
    dup2(fd, 2);
  }
  else
    warning(concat("Could not point standard file descriptors to new /dev/console! (",strerror(errno),")",NULL));
  
  /* Close fd if it's not one of the standard descriptors. */
  if (fd>2) close(fd);
  
  /*
    If realinit is prefixed with lfsdir, strip it out.
  */
  if (!nochroot)
  {
    if (strstr(realinit,lfsdir) == realinit)
      realinit += strlen(lfsdir);
  }
  
  argv[0]="/sbin/init";
  execv(realinit,argv);
  warning(concat("Executing real init (",realinit,") failed! (",strerror(errno),")",NULL));
#ifndef NO_SHELL_FALLBACK
  execl("/bin/sh","/sbin/init", NULL);
  warning(concat("Executing fallback shell (/bin/sh) failed! (",strerror(errno),")",NULL));
#endif
  error("Giving up!");
}

/*
  Outputs an error message.
*/
void warning(const char* msg)
{
  if (is_console(1))
  {
    const char* intro = "\r\npre-init: ";
    write(1, intro, strlen(intro));
    write(1, msg, strlen(msg));
    write(1, "\r\n\r\n", 4);
  }
}

/*
  Outputs an error message and then exits (which causes a kernel panic
  when done as actual init process) or goes into an eternal zombie-collecting
  mode (if ONT_PANIC is defined).
*/
void error(const char* msg)
{
  warning(msg);
#ifdef ONT_PANIC
  for(;;) wait(NULL);
#endif  
  _exit(1);
}

/*
  Returns the concatenation of the input strings. 
  NULL has to be passed as last argument.
  If you think that this function is cool, you have A LOT to learn about
  C programming and I hope that you're not maintainer of some important
  C program right now.
*/
char* concat(const char* str1,...)
{
  ssize_t len = strlen(str1);
  va_list vl;
  
  /*
    Determine the length of the concatenated string.
  */
  va_start(vl,str1);
  const char* str = va_arg(vl,const char*);
  while(str != NULL)
  {
    len += strlen(str);
    str = va_arg(vl,const char*);
  }
  va_end(vl);
  
  /*
    Reserve memory for the concatenated string (don't forget 0-terminator).
  */
  char* newstr = malloc(len+1);
  if ( newstr == NULL ) error("Hah! Hah! Hah!");
  
  /*
    Do the actual concatenation in the most inefficient way possible.
  */
  *newstr = 0;
  strcat(newstr, str1);
  va_start(vl,str1);
  str = va_arg(vl,const char*);
  while(str != NULL)
  {
    strcat(newstr, str);
    str = va_arg(vl,const char*);
  }
  va_end(vl);
  return newstr;
}

/*
  Returns true if file descriptor fd refers to console device
*/
int is_console(int fd)
{
  struct stat buf;
  if (fstat(fd,&buf)<0) return 0;
  return (S_ISCHR(buf.st_mode) 
          && major(buf.st_rdev)==DEV_CONSOLE_MAJOR 
          && minor(buf.st_rdev)==DEV_CONSOLE_MINOR
         );
}

/*
  Performs bind-mounts. 
  See comment at the beginning for the syntax of <bindmount>.
*/
void do_bindmounts(const char* lfsdir, char* bindmount)
{
  char* dest;
  char* source;
  for(;;)
  {
    /*
      Get next source and dest paths.
    */
    dest = bindmount;
    if (dest == NULL) break; 
    
    source = strstr(dest,"=");
    if (source == NULL) break;
    *source = 0;
    source += 1; /* length of "=" */
    
    bindmount = strstr(source,",");
    if (bindmount != NULL)
    {
      *bindmount = 0;
      ++bindmount;
    }
    
    /*
      Relative paths are interpreted relative to <lfsdir>.
    */
    if (*dest != '/')
      dest = concat(lfsdir,"/",dest,NULL);
    if (*source != '/')
      source = concat(lfsdir,"/",source,NULL);
    
    warning(concat("Binding ",source," on top of ",dest,".",NULL));
    if (mount(source, dest, "none", MS_BIND|MS_MGC_VAL, NULL) < 0)
      error(concat("Binding failed! (",strerror(errno),")",NULL));
  }
};

